<?php

namespace App\Http\Controllers;

use App\Aggregator;
use App\Cooks;
use App\Exports\FeedingExport;
use App\Feeding;
use App\Helpers\StaticData;
use App\Imports\FeedingImport;
use App\Notifications\NewFeedingNotification;
use App\School;
use App\State;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use PDF;

class FeedingController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:feeding-list|feeding-create|feeding-edit|feeding-delete', ['only' => ['index','store']]);
        $this->middleware('permission:feeding-create', ['only' => ['create','store']]);
        $this->middleware('permission:feeding-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:feeding-delete', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if(auth()->user()->hasRole('Admin')){
            $feedings  = Feeding::orderBy('id', 'desc')->get();
        }else{
            $feedings  = Feeding::where('manager_id', Auth::user()->id)->orderBy('id', 'desc')->get();
        }
        return view('feeding.index', compact('feedings'));
    }


    public function pupils_request(){
        $feedings  = Feeding::orderBy('id', 'DESC')->where('status', '=', 'Requested')->get();
        return view('feeding.feeding_request', compact('feedings'));
    }

    public function feeding_report_request(){
           if(auth()->user()->hasrole('Admin')){
            $feedings = Feeding::whereIn('status', ['Requested', 'Unpaid'])->orderBy('feeding_date', 'desc')->get();
           }else{
            $feedings = Feeding::where('manager_id', Auth::id())->whereIn('status', ['Requested', 'Unpaid'])->orderBy('feeding_date', 'desc')->get();
           }
            return view('feeding.feeding_report_request', compact('feedings'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user_id = Auth::user()->id;

        $program_managers = User::where('user_type', '=', 'State-Manager')->pluck('name', 'id');

        $states = State::pluck('name', 'id');

        $state = State::get_state_id(Auth::user()->state);
        if(auth()->user()->hasrole('Admin')){
            $schools = School::where('status', '=', 'Approved')->pluck('name', 'id');
        }else{
            $schools = School::where('state', $state)->where('status', '=', 'Approved')->pluck('name', 'id');
        }

        if(auth()->user()->hasrole('Admin')){
            $cooks = Cooks::all()->where('status', '=', 'Approved')->pluck('full_name', 'id');
        }else{
            $cooks = Cooks::all()->where('user_id', $user_id)->where('status', '=', 'Approved')->pluck('full_name', 'id');
        }

        if(auth()->user()->hasrole('Admin')){
            $aggregators = Aggregator::where('status', '=', 'Approved')->pluck('company_name', 'id');
        }else{
            $aggregators = Aggregator::where('program_manager_id', $user_id)->where('status', '=', 'Approved')->pluck('company_name', 'id');
        }

        $weeks = StaticData::sevenDays();
        return view('feeding.create', compact('program_managers', 'cooks', 'aggregators', 'schools','weeks', 'states'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $input = $request->all();
        if(Auth::user()->user_type == 'Admin'){
              $manager_id = $input['manager_id'];
        }else{
            $manager_id = Auth::user()->id;
        }
        $input['manager_id'] = $manager_id;
        $input['status'] = 'Unpaid';

        try{
              $feeding = Feeding::create($input);

              $message = 'New Pupils Added, Payment Status '.$feeding->status;

              // get roles wise users
            $users = User::whereHas(
                'roles', function($q){
                    $q->whereIn('name', ['Admin', 'State-Manager']);
                }
            )->get();

            foreach ($users as $user) {
                $user->notify(new NewFeedingNotification($message));
            }


              return redirect()->back()->with('success', 'Feeding successfully created.');
        }catch(Exception $e){
              return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Feeding  $feeding
     * @return \Illuminate\Http\Response
     */
    public function show(Feeding $feeding)
    {
        return view('feeding.show', compact('feeding'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Feeding  $feeding
     * @return \Illuminate\Http\Response
     */
    public function edit(Feeding $feeding)
    {
        $user_id = Auth::user()->id;

        $states = State::pluck('name', 'id');

        $program_managers = User::where('user_type', '=', 'State-Manager')->pluck('name', 'id');

        $state = State::get_state_id(Auth::user()->state);
        if(auth()->user()->hasrole('Admin')){
            $schools = School::where('status', '=', 'Approved')->pluck('name', 'id');
        }else{
            $schools = School::where('state', $state)->where('status', '=', 'Approved')->pluck('name', 'id');
        }

        if(auth()->user()->hasrole('Admin')){
            $cooks = Cooks::all()->where('status', '=', 'Approved')->pluck('full_name', 'id');
        }else{
            $cooks = Cooks::all()->where('user_id', $user_id)->where('status', '=', 'Approved')->pluck('full_name', 'id');
        }

        if(auth()->user()->hasrole('Admin')){
            $aggregators = Aggregator::where('status', '=', 'Approved')->pluck('company_name', 'id');
        }else{
            $aggregators = Aggregator::where('program_manager_id', $user_id)->where('status', '=', 'Approved')->pluck('company_name', 'id');
        }
        $weeks = StaticData::sevenDays();
        return view('feeding.edit', compact('program_managers', 'cooks', 'aggregators', 'schools','weeks', 'feeding', 'states'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Feeding  $feeding
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Feeding $feeding)
    {
        $input = $request->all();
        try{
              $feeding->update($input);
              return redirect()->back()->with('success', 'Feeding successfully updated.');
        }catch(Exception $e){
              return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Feeding  $feeding
     * @return \Illuminate\Http\Response
     */
    public function destroy(Feeding $feeding)
    {
        try{
            $feeding->delete();
            return redirect()->back()->with('success', 'Feeding successfully deleted.');
      }catch(Exception $e){
            return redirect()->back()->with('error', $e->getMessage())->withInput();
      }
    }

    public function exportFeeding(){
        return Excel::download(new FeedingExport, Carbon::now()->toDateString(). '-' .'pupils-reports.xlsx');
    }

    public function importFeeding(Request $request){
         $validator = Validator::make($request->all(), [
            'select_file'  => 'required|mimes:xls,xlsx'
           ]);

        if($validator->fails()){
            return redirect()->back()->withErrors($validator)->withInput();
        }

          Excel::import(new FeedingImport, $request->file('select_file'));

           return redirect()->back()->with('success', 'Excel Data Imported successfully.');

    }
    // get state wise feeding
    public function get_state_wise_feeding($state){
           $feedings = Feeding::where('state', $state)->orderBy('id', 'desc')->get();
           return view('feeding.state_wise_feeding', compact('feedings'));
    }
}
