<?php

namespace App\Http\Controllers;

use App\Helpers\ImageUpload;
use App\Post;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class PostController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:post-list|post-create|post-edit|post-delete', ['only' => ['index','store']]);
        $this->middleware('permission:post-create', ['only' => ['create','store']]);
        $this->middleware('permission:post-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:post-delete', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
       
      
        if(auth()->user()->hasRole('Admin')){
            $posts = Post::orderBy('id','desc')->get();
        }else{
            $posts = Post::where('user_id', Auth::user()->id)->orderBy('id', 'desc')->get();
        }
        return view('post.index',compact('posts'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('post.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator=Validator::make($request->all(),[
            'thumbnail'=> 'mimes:jpeg,jpg,png,ico,JPG,webp|max:2048',
        ]);
        if($validator->fails()){
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $input=$request->all();

        $input['user_id']  = Auth::user()->id;


        if ($request->hasFile('thumbnail')) {
            $file=$request->file('thumbnail');
            $input['thumbnail']=ImageUpload::image_upload($file);
        }
        try {
            $post=Post::create($input);
            return redirect()->back()->with('success', 'Data has been added.');
        }
        catch (Exception $e) {
            return redirect()->back()->with('error',$e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function show(Post $post)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function edit(Post $post)
    {
        return view('post.edit',compact('post'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Post $post)
    {
        $validator=Validator::make($request->all(),[
            'thumbnail'=> 'mimes:jpeg,bmp,png,jpg,webp|max:2048',
        ]);

        if($validator->fails()){
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $input=$request->all();

        if($request->hasFile('thumbnail')){
            $file=$request->file('thumbnail');
            $input['thumbnail']=ImageUpload::image_upload($file);
            ImageUpload::image_delete($post['thumbnail']);
        }

        try {
            $post->update($input);
            return redirect()->back()->with('success', 'Data has been updated.');
        }
        catch(Exception $e){
              return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Post  $post
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
          $post = Post::findOrFail($id);
          ImageUpload::image_delete($post['thumbnail']); 
          $post->delete();
          return redirect()->back()->with('success', 'Data has been deleted!');
    }

    // post/blog details page
    public function post_details($slug){
           $blog = Post::where('slug', '=', $slug)->first();
           $recent_posts = Post::where('status', '=', 'published')->limit(10)->get();
            // load slider in blog details page
           $slider_posts = Post::where('post_type', '=', 'slider')->limit(5)->get();
           $blogs = Post::where('status', '=', 'published')->orderBy('id', 'DESC')->limit(21)->get();
           return view('blog_details', compact('blog', 'recent_posts', 'slider_posts', 'blogs'));
    }

}
